// Copyright 2014 Google Inc. All Rights Reserved.

#include "Transport.h"

bool Transport::start() {
   bool ret = mReaderThread.start();
   mReaderThread.setPriority(TRANSPORT_GALREADERTHREAD_PRIO);
   ret |= mWriterThread.start();
   mWriterThread.setPriority(TRANSPORT_GALWRITERTHREAD_PRIO);
   return ret;
}

void Transport::requestStop() {
    mStopping = true;
    mReceiver->prepareShutdown();
    abortReads();
}

void Transport::waitForExit() {
    mReaderThread.join();
    LOG("Reader thread joined");
    mWriterThread.join();
    LOG("Writer thread joined");
}

int Transport::notifyStatus(GalReceiverTransportStatus inStatus)
{
    int ret = 0;

    switch(inStatus)
    {
        case GalReceiverTransportStatus::WRITE_SUCCESS:
        {
            /* notify success only when an error occurred previously */
            if (mWriteState != inStatus) {
                mWriteIoErrCnt = 0;
                mWriteState = inStatus;
                if (mCallbacks != nullptr)
                    mCallbacks->notifyStatusCallback(inStatus);
            }
            break;
        }
        case GalReceiverTransportStatus::WRITE_ERR_FATAL:
        case GalReceiverTransportStatus::WRITE_ERR_TIMEOUT:
        {
            mWriteState = inStatus;
            /* timeout and other errors are notified everytime */
            if (mCallbacks != nullptr)
                mCallbacks->notifyStatusCallback(inStatus);
            break;
        }
        case GalReceiverTransportStatus::WRITE_ERR_IO:
        {
            mWriteState = inStatus;
            /* I/O error can occur more frequently and will be notified
             * after 10 occurrences */
            if (mWriteIoErrCnt == 0) {
                if (mCallbacks != nullptr)
                    mCallbacks->notifyStatusCallback(inStatus);
            }
            mWriteIoErrCnt++;
            if (mWriteIoErrCnt >= 10) {
                mWriteIoErrCnt = 0;
            }
            break;
        }
        case GalReceiverTransportStatus::READ_SUCCESS:
        {
            /* notify success only when an error occurred previously */
            if (mReadState != inStatus) {
                mReadIoErrCnt = 0;
                mReadState = inStatus;
                if (mCallbacks != nullptr)
                    mCallbacks->notifyStatusCallback(inStatus);
            }
            break;
        }
        case GalReceiverTransportStatus::READ_ERR_FATAL:
        case GalReceiverTransportStatus::READ_ERR_TIMEOUT:
        {
            mReadState = inStatus;
            /* timeout and other errors are notified everytime */
            if (mCallbacks != nullptr)
                mCallbacks->notifyStatusCallback(inStatus);
            break;
        }
        case GalReceiverTransportStatus::READ_ERR_IO:
        {
            mReadState = inStatus;
            /* I/O error can occur more frequently and will be notified
             * after 10 occurrences */
            if (mReadIoErrCnt == 0) {
                if (mCallbacks != nullptr)
                    mCallbacks->notifyStatusCallback(inStatus);
            }
            mReadIoErrCnt++;
            if (mReadIoErrCnt >= 10) {
                mReadIoErrCnt = 0;
            }
            break;
        }
        default:
        {
            LOGW("Notified unknown status/error (%d).", (int)inStatus);
            if (mCallbacks != nullptr)
                mCallbacks->notifyStatusCallback(GalReceiverTransportStatus::UNKNOWN_ERR);
            break;
        }
    }

    return ret;
}

bool Transport::readOrFail(void* buf, size_t len) {
    int bytesRemaining = len;
    uint8_t* ptr = (uint8_t*) buf;
    while (bytesRemaining > 0 && !mStopping) {
        int ret = read(ptr, bytesRemaining);
        if (ret == TRANSPORT_ERROR_FATAL) {
            LOGE("Encounted fatal error on read.");
            return false;
        } else if (ret == TRANSPORT_ERROR_NONFATAL) {
            // Keep going till it succeeds or a stop is requested.
            continue;
        }
        ptr += ret;
        bytesRemaining -= ret;
    }

    // Still need to return false in the mStopping case.
    return bytesRemaining == 0;
}

bool Transport::writeOrFail(void* buf, size_t len) {
    int bytesRemaining = len;
    uint8_t* ptr = (uint8_t*) buf;
    while (bytesRemaining > 0 && !mStopping) {
        int ret = write(ptr, bytesRemaining);
        if (ret == TRANSPORT_ERROR_FATAL) {
            LOGE("Encounted fatal error on write.");
            return false;
        } else if (ret == TRANSPORT_ERROR_NONFATAL) {
            // Keep going till it succeeds or a stop is requested.
            continue;
        }
        ptr += ret;
        bytesRemaining -= ret;
    }

    // Still need to return false in the mStopping case.
    return bytesRemaining == 0;
}

void GalReaderThread::run() {
    uint8_t header[FRAME_HEADER_MIN_LENGTH];
    while (!mTransport->mStopping) {
        bool ret = mTransport->readOrFail(header, sizeof(header));
        // If we are stopping we can afford to throw away the data anyway.
        if (mTransport->mStopping) {
            LOG("Reader thread shutting down on request.");
            break;
        }

        if (ret == false) {
            break;
        }

        int len = mTransport->mReceiver->getAdditionalBytesToRead(header);
        if (len <= 0) {
            // Something is horribly broken and the length we read out of the header
            // was negative. Log an error and abort.
            LOGE("Bad payload length %d. Shutting down due to potential framing error.", len);
            ret = false;
            break;
        }

        size_t total = len + sizeof(header);
        shared_ptr<IoBuffer> buf = mTransport->mReceiver->allocateBuffer(total);
        memcpy(buf->raw(), header, sizeof(header));
        ret = mTransport->readOrFail((uint8_t*) buf->raw() + sizeof(header), len);
        /* changed by ADIT
         * If we are stopping we can afford to throw away the data anyway. */
        if (mTransport->mStopping) {
            LOG("Reader thread shutting down on request.");
            break;
        }

        if (ret == false) {
            break;
        }
        mTransport->mReceiver->queueIncoming(buf);
        // Don't touch buf after this.
    }

    /* changed by ADIT
     * Call unrecoverableError() to inform upper layer about error.
     * Expect that messageRouter points to a static member. */
    if (mTransport->mStopping == false) {
        mTransport->mReceiver->messageRouter()->unrecoverableError(STATUS_FRAMING_ERROR);
        mTransport->requestStop();
    }

    LOG("Reader thread exiting.");
}

void GalWriterThread::run() {
    IoBuffer buf;
    while (!mTransport->mStopping) {
        bool ret = mTransport->mReceiver->getEncodedFrame(&buf);
        if (ret && !mTransport->mStopping) {
            ret = mTransport->writeOrFail(buf.raw(), buf.size());
            if (mTransport->mStopping) {
                LOG("Writer thread shutting down on request.");
                break;
            }

            if (ret == false) {
                break;
            }
        }
    }

    if (mTransport->mStopping == false) {
        /* changed by ADIT
         * Call unrecoverableError() to inform upper layer about error.
         * Expect that messageRouter points to a static member. */
        mTransport->mReceiver->messageRouter()->unrecoverableError(STATUS_FRAMING_ERROR);
        mTransport->requestStop();
    }

    LOG("Writer thread exiting.");
}
